/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.ad.caching;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.apache.commons.lang.builder.EqualsBuilder;
import org.apache.commons.lang.builder.HashCodeBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.ad.ExpiringState;
import org.opensearch.ad.MemoryTracker;
import org.opensearch.ad.caching.PriorityTracker;
import org.opensearch.ad.ml.EntityModel;
import org.opensearch.ad.ml.ModelState;
import org.opensearch.ad.model.InitProgressProfile;
import org.opensearch.ad.ratelimit.CheckpointWriteWorker;
import org.opensearch.ad.ratelimit.RequestPriority;

public class CacheBuffer
implements ExpiringState {
    private static final Logger LOG = LogManager.getLogger(CacheBuffer.class);
    private final int MAX_TRACKING_ENTITIES = 1000000;
    private int minimumCapacity;
    private final ConcurrentHashMap<String, ModelState<EntityModel>> items;
    private final long memoryConsumptionPerEntity;
    private final MemoryTracker memoryTracker;
    private final Duration modelTtl;
    private final String detectorId;
    private Instant lastUsedTime;
    private long reservedBytes;
    private final PriorityTracker priorityTracker;
    private final Clock clock;
    private final CheckpointWriteWorker checkpointWriteQueue;
    private final Random random;

    public CacheBuffer(int minimumCapacity, long intervalSecs, long memoryConsumptionPerEntity, MemoryTracker memoryTracker, Clock clock, Duration modelTtl, String detectorId, CheckpointWriteWorker checkpointWriteQueue, Random random) {
        this.memoryConsumptionPerEntity = memoryConsumptionPerEntity;
        this.setMinimumCapacity(minimumCapacity);
        this.items = new ConcurrentHashMap();
        this.memoryTracker = memoryTracker;
        this.modelTtl = modelTtl;
        this.detectorId = detectorId;
        this.lastUsedTime = clock.instant();
        this.clock = clock;
        this.priorityTracker = new PriorityTracker(clock, intervalSecs, clock.instant().getEpochSecond(), 1000000);
        this.checkpointWriteQueue = checkpointWriteQueue;
        this.random = random;
    }

    private void update(String entityModelId) {
        this.priorityTracker.updatePriority(entityModelId);
        Instant now = this.clock.instant();
        this.items.get(entityModelId).setLastUsedTime(now);
        this.lastUsedTime = now;
    }

    public void put(String entityModelId, ModelState<EntityModel> value) {
        this.put(entityModelId, value, value.getPriority());
    }

    private void put(String entityModelId, ModelState<EntityModel> value, float priority) {
        ModelState<EntityModel> contentNode = this.items.get(entityModelId);
        if (contentNode == null) {
            this.priorityTracker.addPriority(entityModelId, priority);
            this.items.put(entityModelId, value);
            Instant now = this.clock.instant();
            value.setLastUsedTime(now);
            this.lastUsedTime = now;
            if (!this.sharedCacheEmpty()) {
                this.memoryTracker.consumeMemory(this.memoryConsumptionPerEntity, false, MemoryTracker.Origin.HC_DETECTOR);
            }
        } else {
            this.update(entityModelId);
            this.items.put(entityModelId, value);
        }
    }

    public ModelState<EntityModel> get(String key) {
        ModelState<EntityModel> node = this.items.get(key);
        if (node == null) {
            return null;
        }
        this.update(key);
        return node;
    }

    public boolean canRemove() {
        return !this.items.isEmpty() && this.items.size() > this.minimumCapacity;
    }

    public ModelState<EntityModel> remove() {
        Optional<String> key = this.priorityTracker.getMinimumPriorityEntityId();
        if (key.isPresent()) {
            return this.remove(key.get());
        }
        return null;
    }

    public ModelState<EntityModel> remove(String keyToRemove) {
        this.priorityTracker.removePriority(keyToRemove);
        boolean reserved = this.sharedCacheEmpty();
        ModelState<EntityModel> valueRemoved = this.items.remove(keyToRemove);
        if (valueRemoved != null) {
            EntityModel modelRemoved;
            if (!reserved) {
                this.memoryTracker.releaseMemory(this.memoryConsumptionPerEntity, false, MemoryTracker.Origin.HC_DETECTOR);
            }
            if ((modelRemoved = valueRemoved.getModel()) != null) {
                boolean isNullModel = !modelRemoved.getTrcf().isPresent();
                this.checkpointWriteQueue.write(valueRemoved, isNullModel, RequestPriority.MEDIUM);
                modelRemoved.clear();
            }
        }
        return valueRemoved;
    }

    public boolean dedicatedCacheAvailable() {
        return this.items.size() < this.minimumCapacity;
    }

    public boolean sharedCacheEmpty() {
        return this.items.size() <= this.minimumCapacity;
    }

    public long getMemoryConsumptionPerEntity() {
        return this.memoryConsumptionPerEntity;
    }

    public boolean canReplaceWithinDetector(float priority) {
        if (this.items.isEmpty()) {
            return false;
        }
        Optional<Map.Entry<String, Float>> minPriorityItem = this.priorityTracker.getMinimumPriority();
        return minPriorityItem.isPresent() && priority > minPriorityItem.get().getValue().floatValue();
    }

    public ModelState<EntityModel> replace(String entityModelId, ModelState<EntityModel> value) {
        ModelState<EntityModel> replaced = this.remove();
        this.put(entityModelId, value);
        return replaced;
    }

    public List<ModelState<EntityModel>> maintenance() {
        ArrayList<ModelState<EntityModel>> modelsToSave = new ArrayList<ModelState<EntityModel>>();
        ArrayList<ModelState<EntityModel>> removedStates = new ArrayList<ModelState<EntityModel>>();
        this.items.entrySet().stream().forEach(entry -> {
            String entityModelId = (String)entry.getKey();
            try {
                ModelState modelState = (ModelState)entry.getValue();
                Instant now = this.clock.instant();
                if (modelState.getLastUsedTime().plus(this.modelTtl).isBefore(now)) {
                    removedStates.add(this.remove(entityModelId));
                } else if (this.random.nextInt(6) == 0) {
                    modelsToSave.add(modelState);
                }
            }
            catch (Exception e) {
                LOG.warn("Failed to finish maintenance for model id " + entityModelId, (Throwable)e);
            }
        });
        this.checkpointWriteQueue.writeAll(modelsToSave, this.detectorId, false, RequestPriority.MEDIUM);
        return removedStates;
    }

    public int getActiveEntities() {
        return this.items.size();
    }

    public boolean isActive(String entityModelId) {
        return this.items.containsKey(entityModelId);
    }

    public long getLastUsedTime(String entityModelId) {
        ModelState<EntityModel> state = this.items.get(entityModelId);
        if (state != null) {
            return state.getLastUsedTime().toEpochMilli();
        }
        return -1L;
    }

    public Optional<EntityModel> getModel(String entityModelId) {
        return Optional.of(this.items).map(map -> (ModelState)map.get(entityModelId)).map(state -> (EntityModel)state.getModel());
    }

    public void clear() {
        this.memoryTracker.releaseMemory(this.getReservedBytes(), true, MemoryTracker.Origin.HC_DETECTOR);
        if (!this.sharedCacheEmpty()) {
            this.memoryTracker.releaseMemory(this.getBytesInSharedCache(), false, MemoryTracker.Origin.HC_DETECTOR);
        }
        this.items.clear();
        this.priorityTracker.clearPriority();
    }

    public long getReservedBytes() {
        return this.reservedBytes;
    }

    public long getBytesInSharedCache() {
        int sharedCacheEntries = this.items.size() - this.minimumCapacity;
        if (sharedCacheEntries > 0) {
            return this.memoryConsumptionPerEntity * (long)sharedCacheEntries;
        }
        return 0L;
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        if (obj instanceof InitProgressProfile) {
            CacheBuffer other = (CacheBuffer)obj;
            EqualsBuilder equalsBuilder = new EqualsBuilder();
            equalsBuilder.append((Object)this.detectorId, (Object)other.detectorId);
            return equalsBuilder.isEquals();
        }
        return false;
    }

    public int hashCode() {
        return new HashCodeBuilder().append((Object)this.detectorId).toHashCode();
    }

    @Override
    public boolean expired(Duration stateTtl) {
        return this.expired(this.lastUsedTime, stateTtl, this.clock.instant());
    }

    public String getDetectorId() {
        return this.detectorId;
    }

    public List<ModelState<EntityModel>> getAllModels() {
        return this.items.values().stream().collect(Collectors.toList());
    }

    public PriorityTracker getPriorityTracker() {
        return this.priorityTracker;
    }

    public void setMinimumCapacity(int minimumCapacity) {
        if (minimumCapacity < 0) {
            throw new IllegalArgumentException("minimum capacity should be larger than or equal 0");
        }
        this.minimumCapacity = minimumCapacity;
        this.reservedBytes = this.memoryConsumptionPerEntity * (long)minimumCapacity;
    }
}

